#!/usr/bin/python

#
# Python Traffic Analyzer (PyTrAn)
#

import re
import os
import sys
import time
import random

class pytran:
    ############################################################################
    ### constructor
    ###
    def __init__(self, camera_id, window_id, refresh_rate, color_threshold):
        self.camera_id       = camera_id
        self.window_id       = window_id
        self.refresh_rate    = refresh_rate
        self.color_threshold = color_threshold

        self.this = "pytran.this"
        self.last = "pytran.last"
        self.diff = "pytran.diff"

        self.diff_ascii = "pytran.diff.ascii"
        self.mask_ascii = "mask_" + str(self.camera_id) + ".ascii"

        self.debug_flag = False

        self.debug("__init__")

        # set the banner
        self.banner = "Python Traffic Analyzer (PyTrAn)"

        # this is a unix only script. we detect if we are in windows by
        # attempting to access a module that only exists in unix.
        try:
            os.getuid()
        except:
            print
            print "could not initialize. pytran does not work in win32."
            sys.exit()


    ############################################################################
    ### getters
    ###
    def get_banner          (self): return self.banner
    def get_camera_id       (self): return self.camera_id
    def get_color_threshold (self): return self.color_threshold
    def get_debug_flag      (self): return self.debug_flag
    def get_diff            (self): return self.diff
    def get_diff_ascii      (self): return self.diff_ascii
    def get_last            (self): return self.last
    def get_mask_ascii      (self): return self.mask_ascii
    def get_refresh_rate    (self): return self.refresh_rate
    def get_this            (self): return self.this
    def get_window_id       (self): return self.window_id


    ############################################################################
    ### setters
    ###
    def set_camera_id       (self, value): self.camera_id       = value
    def set_color_threshold (self, value): self.color_threshold = value
    def set_debug_flag      (self, value): self.debug_flag      = value
    def set_diff            (self, value): self.diff            = value
    def set_diff_ascii      (self, value): self.diff_ascii      = value
    def set_last            (self, value): self.last            = value
    def set_mask_ascii      (self, value): self.mask_ascii      = value
    def set_refresh_rate    (self, value): self.refresh_rate    = value
    def set_this            (self, value): self.this            = value
    def set_window_id       (self, value): self.window_id       = value


    ############################################################################
    ### debug()
    ###
    def debug(self, msg):
        if self.debug_flag:
            print "DEBUG> " + msg


    ############################################################################
    ### display()
    ###
    ### displays an image to the specified window id.
    def display(self, img):
        self.debug("displaying image: " + img)

        os.system("display -window "
                + self.window_id
                + " "
                + img)


    ############################################################################
    ### grab_frame()
    ###
    ### grabs a frame from self.camera_id and stores it as the "this" frame.
    def grab_frame(self):
        self.debug("grabbing frame from camera " + str(self.camera_id))

        os.system("curl http://cam01.trafficland.com/camera/"
                + str(self.camera_id)
                + "/live.jpg?"
                + str(random.random()) # prevent caching
                + " 2>/dev/null | djpeg > "
                + self.this)


    ############################################################################
    ### gen_diff()
    ###
    ### generates the pnmarith difference between two captures frames.
    def gen_diff(self):
        self.debug("generating frame diff on " + self.last + ", " + self.this)

        os.system("pnmarith -diff "
                + self.last
                + " "
                + self.this
                + " > "
                + self.diff)


    ############################################################################
    ### noraw()
    ###
    ### converts the binary diff frame to ascii.
    ###
    def noraw(self):
        self.debug("converting " + self.diff + " to ascii")

        os.system("pnmnoraw "
                + self.diff
                + " > "
                + self.diff + ".ascii")


    ############################################################################
    ### ratio()
    ###
    ### calculates the ratio of bright pixels over the mask.
    ###
    ### returns: floating point ratio.
    def ratio(self):
        num_pix_diff = 0
        num_pix_mask = 0

        self.debug("calculating traffic ratio...")

        try:
            f_mask = file(self.mask_ascii, 'r')
            f_diff = file(self.diff_ascii, 'r')
        except:
            raise Exception

        # read off the first 3 lines.
        # these specify various image parameters that we don't care about.
        for n in range(3):
            try:
                f_mask.readline()
                f_diff.readline()
            except:
                raise Exception

        # loop through the rest of the file line by line.
        for l_diff in f_diff.readlines():
            try:
                l_mask = f_mask.readline()
            except:
                raise Exception

            # split the line into an array of per pixel RGB values.
            s_diff = l_diff.split("  ")
            s_mask = l_mask.split("  ")

            # take the red portion of the RGB value and.
            for i in range(len(s_diff)):
                s_diff[i] = int(re.sub("\s.*", "", s_diff[i]))
                s_mask[i] = int(re.sub("\s.*", "", s_mask[i]))

                # if this pixel is active in the mask, count it.
                if s_mask[i]:
                    num_pix_mask += 1
                    # if this pixel is greater then the defined threshold,
                    # count it.
                    if s_diff[i] > self.color_threshold:
                         num_pix_diff += 1

        # prevent divide by zero.
        if num_pix_mask == 0:
            raise Exception

        # return the ratio of bright pixels over the mask.
        return float(num_pix_diff) / float(num_pix_mask)


    ############################################################################
    ### refresh()
    ###
    ### sleeps for an allotted time.
    def refresh(self):
        self.debug("refreshing image in " + str(self.refresh_rate) + " secs")

        try:
            time.sleep(self.refresh_rate)
        except:
            sys.exit()


    ############################################################################
    ### rotate()
    ###
    ### moves the current frame over the last one.
    def rotate(self):
        self.debug("rotating image: " + self.this + " > " + self.last)

        os.system("mv -f "
                + self.this
                + " "
                + self.last)


################################################################################


if __name__ == '__main__':

    camera_id       = 220
    window_id       = "0x1600011"
    refresh_rate    = 3
    color_threshold = 15

    p = pytran(camera_id, window_id, refresh_rate, color_threshold)

    print p.get_banner()
    print

    #p.set_debug_flag(True)
    p.grab_frame()
    p.rotate()
    p.refresh()

    start   = int(time.time())
    samples = 0
    total   = 0

    print "taking a 5 minute sample."

    while (int(time.time()) < start + 60 * 5):
        try:
            p.grab_frame()
            p.gen_diff()
            p.display(p.get_diff())
            p.noraw()
            ratio = p.ratio() * 100
            if ratio > 0:
                samples += 1
                total   += ratio
                print "ratio: %d%%    \r" % (ratio),
                sys.stdout.flush()
            p.rotate()
            p.refresh()
        except:
            sys.exit()

    print
    print "5 minute sample: %.02f" % (total / samples)
    print
