/******************************************************************************
 *  [ 1984 ]                                                                  *
 *                                                                            *
 *  pedram amini (pedram@redhive.com)                                         *
 *  pedram.redhive.com                                                        *
 *                                                                            *
 ******************************************************************************/

#include "1984.h"

int
main (int argc, char **argv)    {
    int  i,
         opt,
         cl_length = 0;              /* command line length            */
    char *packet,                    /* captured packet                */
         *device = NULL,             /* interface to sniff on          */
         filter[1024],               /* tcpdump style capture filter   */
         errbuf[PCAP_ERRBUF_SIZE];   /* error buffer for pcap          */
    pid_t pid;                       /* used in daemonizing            */
    struct pcap_pkthdr pcap_h;       /* pcap packet header             */
    pcap_t *capdev;                  /* the capture device             */
    struct _options options;         /* 1984 options data struct       */

    /* determine command line length (so we can clear it later */
    for (i = 0; i < argc; i++)
        cl_length += strlen(argv[i]) + 1;

    memset(&options, '\0', sizeof(struct _options));

    /* parse the command line */
    while ((opt = getopt(argc, argv, "b:dhi:psv")) != EOF)    {
        switch (opt)  {
            case 'b':               /* base log directory */
                strncpy(options.base_log_dir, optarg, OPT_DIR);
                break;
            case 'd':               /* daemonize */
                options.flag_daemonize++;
                break;
            case 'i':               /* interface */
                device = optarg;
                break;
            case 'p':               /* report statistics through argv[0]*/
                options.flag_ps_report++;
                options.flag_statistics++;
                break;
            case 's':               /* log statistics */
                options.flag_statistics++;
                break;
            case 'v':               /* verbose */
                options.flag_verbose++;
                break;
            case 'h':               /* usage */
            case '?':
                usage(argv);
                break;
        }
    }

    /* default values */
    if (strlen(options.base_log_dir) == 0)
        strncpy(options.base_log_dir, BASE_LOG_DIR, OPT_DIR);

    /* make sure there is a device to sniff on */
    if (!device)
        device = pcap_lookupdev(errbuf);

    if (!device) {
        fprintf(stderr, "\ndevice lookup failed");
        exit (EXIT_FAILURE);
     }

    /* set capture filter. */
    argc -= optind;
    argv += optind;

    strcpy(filter, TCPDUMP_FILTER);
    if (argc != 0)  {
        strcat(filter, " and ");
        strcat(filter, copy_argv(argv));
    }

    argc += optind;
    argv -= optind;

    /* prepare the device for capturing */
    capdev = set_cap_dev(device, filter, &options);

    /* print some informative information */
    printf("\n[ 1984 %s ]\n", VERSION);
    printf("\nsniffing on:          %s", device);
    printf("\nusing filter:         %s", filter);
    printf("\nlogging to:           %s", options.base_log_dir);
    printf("\nverbose output:       %s", options.flag_verbose    ? "on":"off");
    printf("\nstatistics tracking:  %s", options.flag_statistics ? "on":"off");
    printf("\n");

    /* daemonize */
    if (options.flag_daemonize)  {
        pid = fork();

        if (pid < 0) {
            printf("\ncould not daemonize");
            exit (EXIT_FAILURE);
        }

        if (pid != 0)   {
            printf("\ndaemonizing...\n");
            exit (EXIT_SUCCESS);
        }

        /* no use in being verbose anymore */
        options.flag_verbose = 0;

        fclose(stdin); fclose(stdout); fclose(stderr);
        setsid();
        umask(0);
    }

    /* clear out a possible lengthy command line for ps output */
    memset(argv[0], '\0', cl_length);
    sprintf(argv[0], "1984");

    /* main loop */
    for (i = 0; ; i++)   {
        packet = (u_char *)pcap_next(capdev, &pcap_h);

        if (packet == NULL)
            continue;

        /* print pretty twirling ascii (disabled) */
        if (0)  {
            switch (i % 4)  {
                case 0: printf("|\b");  break;
                case 1: printf("/\b");  break;
                case 2: printf("-\b");  break;
                case 3: printf("\\\b"); break;
            }
            fflush(NULL);
        }

        parse(packet, &options);

        /* show the current statistics through 'ps' */
        if (options.flag_ps_report)    {
            memset(argv[0], '\0', strlen(argv[0]));
            sprintf(argv[0], "1984 m:%d w:%d c:%d", options.num_messages,
                                                    options.num_words,
                                                    options.num_chars);
        }
    }

    /* not reached */
    return (EXIT_SUCCESS);
}


/******************************************************************************
 * copy_argv (from tcpdump)                                                   *
 *                                                                            *
 *  used for extracting filter string from command line                       *
 *  arg1: (char **) pointer to vector arguments                               *
 *  ret:  (char *)  filter string                                             *
 ******************************************************************************/
char *
copy_argv (char **argv)    {
    char **p, *buf, *src, *dst;
    u_int len = 0;

    p = argv;
    if (*p == 0)
        return 0;

    while (*p)
        len += strlen(*p++) + 1;

    if ((buf = (char *)malloc(len)) == NULL) {
        perror("malloc");
        exit (EXIT_FAILURE);
    }

    p = argv;
    dst = buf;
    while ((src = *p++) != NULL) {
        while ((*dst++ = *src++) != '\0');
        dst[-1] = ' ';
    }
    dst[-1] = '\0';

    return buf;
}


/******************************************************************************
 *  log_message                                                               *
 *                                                                            *
 *  used in logging chat messages.                                            *
 *  arg1: (struct _log_info *) pointer to 1984 log info datas truct           *
 *  arg2: (struct _options  *) pointer to 1984 options data struct            *
 *  ret:  none                                                                *
 ******************************************************************************/
void
log_message (struct _log_info *log_info, struct _options *options)   {

    int    word_count = 0;
    char   log_message[LI_MESSAGE],
           log_path[MAX_PATH_LENGTH],
           date[10], timestamp[5],
           *t;      /* for message traversal */
    FILE   *fstream;
    time_t now;
    struct tm *tm;

    memset(log_message, '\0', LI_MESSAGE);
    memset(log_path,    '\0', MAX_PATH_LENGTH);

    /* create readable date / time strings */
    now = time(0);
    tm  = localtime((time_t *)&now);

    sprintf(date, "%02d.%02d.%d",   tm->tm_mon,
                                    tm->tm_mday,
                                    tm->tm_year + 1900);

    sprintf(timestamp, "%02d:%02d.%02d", tm->tm_hour,
                                         tm->tm_min,
                                         tm->tm_sec);

    /* if statistics flag is on refresh statistics and log them */
    if (options->flag_statistics)    {

        /* count the number of words in the message */
        for (t = log_info->message; *t != '\0'; t++)
            if (*t == ' ')
                word_count++;

        /* update the counts */
        options->num_messages++;
        options->num_words += (word_count + 1);
        options->num_chars += strlen(log_info->message);

        /* write the stats to disk */
        strcat(log_path, options->base_log_dir);
        strcat(log_path, "/statistics.log");

        /* open the log file with the stream positioned to the beginning */
        if ( !(fstream = fopen(log_path, "w+")) )    {
            perror("fopen");
            exit (EXIT_FAILURE);
        }

        /* write the entry */
        fprintf(fstream, "# logged messages:   %d\n", options->num_messages);
        fprintf(fstream, "# logged words:      %d\n", options->num_words);
        fprintf(fstream, "# logged characters: %d\n", options->num_chars);

        /* close log file and cleanup path buffer */
        fclose(fstream);
        memset(log_path, '\0', MAX_PATH_LENGTH);

        /* time to be verbose */
        if (options->flag_verbose)   {
            printf("\ncurrent statistics:");
            printf("\n\t# logged messages:   %d",   options->num_messages);
            printf("\n\t# logged words:      %d",   options->num_words);
            printf("\n\t# logged characters: %d\n", options->num_chars);
        }
    }

    /* construct the log message */
    switch (log_info->direction)  {
        case LI_TO_CLIENT:
            strcat(log_message, "<font face='courier new'><font color=blue>");
            strcat(log_message, log_info->s_name);
            strcat(log_message, "<!--");
            strcat(log_message, log_info->s_addr);
            strcat(log_message, "--><small> (");
            strcat(log_message, timestamp);
            strcat(log_message, ") </small>");
            strcat(log_message, "&gt;</font> ");
            strcat(log_message, log_info->message);
            strcat(log_message, "</font><br>");
            break;
        case LI_FROM_CLIENT:
            strcat(log_message, "<font face='courier new'><font color=red>");
            //strcat(log_message, log_info->d_name);
            strcat(log_message, "<!--");
            strcat(log_message, log_info->s_addr);
            strcat(log_message, "--><small> (");
            strcat(log_message, timestamp);
            strcat(log_message, ") </small>");
            strcat(log_message, "&gt;</font> ");
            strcat(log_message, log_info->message);
            strcat(log_message, "</font><br>");
            break;
    }

    strcat(log_path, options->base_log_dir);
    strcat(log_path, "/");

    switch (log_info->direction)  {
        case LI_FROM_CLIENT:
            strcat(log_path, log_info->s_addr);
            break;
        case LI_TO_CLIENT:
            strcat(log_path, log_info->d_addr);
            break;
    }

    /* ensure the address directory exists */
    mkdir(log_path, LOG_PERMS);

    strcat(log_path, "/");

    /* NEW_PROTO - add a case for new protocols */
    switch (log_info->protocol) {
        case OSCAR:
        case TOC:
            printf("\nAIM> %s %s", date, timestamp);
            strcat(log_path, "aim");
            break;
        case MSN:
            printf("\nMSN> %s %s", date, timestamp);
            strcat(log_path, "msn");
            break;
    }

    /* ensure the protocol directory exists */
    mkdir(log_path, LOG_PERMS);

    strcat(log_path, "/");

    switch (log_info->direction)  {
        case LI_FROM_CLIENT:
            strcat(log_path, log_info->d_name);
            break;
        case LI_TO_CLIENT:
            strcat(log_path, log_info->s_name);
            break;
    }

    /* ensure the screenname subdirectory exists */
    mkdir(log_path, LOG_PERMS);

    strcat(log_path, "/");
    strcat(log_path, date);
    strcat(log_path, ".html");

    /* open the log file and write the entry */
    if ( !(fstream = fopen(log_path, "a+")) )    {
        perror("fopen");
        exit (EXIT_FAILURE);
    }

    fprintf(fstream, "%s\n", log_message);
    fclose(fstream);

    printf("\n%s -> %s", log_info->s_addr, log_info->d_addr);
    printf("\n%s -> %s", log_info->s_name, log_info->d_name);
    printf("\n%s",       log_info->message);
    printf("\n");
}


/******************************************************************************
 *  log_signon                                                                *
 *                                                                            *
 *  used in logging signons.                                                  *
 *  arg1: (struct _log_info *) pointer to 1984 log info data struct           *
 *  arg2: (struct _options  *) pointer to 1984 options data struct            *
 *  ret:  none                                                                *
 ******************************************************************************/
void
log_signon (struct _log_info *log_info, struct _options *options)   {

    char log_message[LI_MESSAGE],
         log_path[MAX_PATH_LENGTH],
         timestamp[20];
    FILE *fstream;
    time_t now;
    struct tm *tm;

    memset(log_message, '\0', LI_MESSAGE);
    memset(log_path,    '\0', MAX_PATH_LENGTH);

    /* create a readable timestamp */
    now = time(0);
    tm  = localtime((time_t *)&now);

    sprintf(timestamp, "%02d.%02d.%d %02d:%02d", tm->tm_mon,
                                                 tm->tm_mday,
                                                 tm->tm_year + 1900,
                                                 tm->tm_hour,
                                                 tm->tm_min);

    /* construct the log message */
    strcat(log_message, timestamp);
    strcat(log_message, " ");
    strcat(log_message, log_info->s_addr);
    strcat(log_message, " signed on as ");
    strcat(log_message, log_info->s_name);

    strcat(log_path, options->base_log_dir);
    strcat(log_path, "/logins");

    /* NEW_PROTO - add a case for new protocols */
    switch (log_info->protocol) {
        case OSCAR:
        case TOC:
            printf("\nAIM> %s", timestamp);
            strcat(log_path, ".aim");
            break;
        case MSN:
            printf("\nMSN> %s", timestamp);
            strcat(log_path, ".msn");
            break;
    }

    /* open the log file and write the entry */
    if ( !(fstream = fopen(log_path, "a+")) )    {
        perror("fopen");
        exit (EXIT_FAILURE);
    }

    fprintf(fstream, "%s\n", log_message);
    fclose(fstream);

    printf("\n%s signed on as %s", log_info->s_addr, log_info->s_name);
    printf("\n");
}


/******************************************************************************
 *  parse                                                                     *
 *                                                                            *
 *  determines a packets protocol and calls the appropriate handler (if any). *
 *  arg1: (char *)            pointer to packet                               *
 *  arg2: (struct _options *) pointer to 1984 options struct                  *
 *  ret:  none                                                                *
 ******************************************************************************/
void
parse (char *packet, struct _options *options)   {
    struct    tcphdr *tcp;
    u_int16_t tcp_source,
              tcp_dest;
    
    tcp = (struct tcphdr *) (packet + options->offset + IP_H);
    
    /* we don't do the byte conversion on tcp->{source, dest} so as not to
       confuse ourselves later */
    tcp_source = tcp->source;
    tcp_dest   = tcp->dest;
    tcp_source = ntohs(tcp_source);
    tcp_dest   = ntohs(tcp_dest);
    
    /* NEW_PROTO - add a case for new protocols */

    if (tcp_source == OSCAR || tcp_dest == OSCAR)
        oscar_parse(packet, options);
        
    if (tcp_source == TOC || tcp_dest == TOC)
        toc_parse(packet, options);
        
    if (tcp_source == MSN || tcp_dest == MSN)
        msn_parse(packet, options);
}


/******************************************************************************
 *  set_cap_dev                                                               *
 *                                                                            *
 *  sniff on appropriate device, set filter, and calculate datalink offset    *
 *  arg1: (char *)             pointer to device name                         *
 *  arg2: (char *)             pointer to filter string                       *
 *  arg3: (struct _options  *) pointer to 1984 options struct                 *
 *  ret:  (pcap_t *)           pointer to pcap device                         *
 ******************************************************************************/
pcap_t *
set_cap_dev (char *device, char *filter, struct _options *options) {
    unsigned int network, netmask;  /* for filter setting    */
    struct bpf_program prog;        /* store compiled filter */
    pcap_t *capdev;                 /* the capture device    */
    char errbuf[PCAP_ERRBUF_SIZE];  /* pcap error buffer     */

    pcap_lookupnet(device, &network, &netmask, errbuf);

    if ((capdev = pcap_open_live(device,
                                 SNAPLEN,
                                 PROMISC,
                                 TO_MS,
                                 errbuf)) == NULL)    {
        perror("pcap_open_live");
        exit (EXIT_FAILURE);
    }

    /* we only want to see traffic specified by filter so compile and set it */
    pcap_compile(capdev, &prog, filter, 0, netmask);
    pcap_setfilter(capdev, &prog);

    /* determine datalink offset */
    switch (pcap_datalink(capdev)) {
        case DLT_EN10MB:
            options->offset = 14;
            break;
        case DLT_IEEE802:
            options->offset = 22;
            break;
        case DLT_FDDI:
            options->offset = 21;
            break;
        case DLT_NULL:
            options->offset = 4;
            break;
        case DLT_RAW:
            options->offset = 0;
            break;
        default:
            fprintf(stderr, "\n%s bad datalink type", device);
            exit (EXIT_FAILURE);
            break;
  }

  return capdev;
}


/******************************************************************************
 *  usage                                                                     *
 *                                                                            *
 *  strip /'s and print program usage                                         *
 *  arg1: (char **) pointer to vector arguments                               *
 *  ret:  none                                                                *
 ******************************************************************************/
void
usage (char **argv)    {
    char *p, *name;

    if ((p = strrchr(argv[0], '/')) != NULL)
        name = p + 1;
    else
        name = argv[0];

    printf("\n[ 1984 %s ]", VERSION);
    printf("\nUsage: %s [options] optional-tcpdump-filter", name);
    printf("\n\t-b <base log directory> defaults to %s", BASE_LOG_DIR);
    printf("\n\t-d daemonize");
    printf("\n\t-h this screen");
    printf("\n\t-i <interface> to sniff on");
    printf("\n\t-p update program name to contain statistics (forces -s)");
    printf("\n\t-s log statistics on captured messages");
    printf("\n\t-v verbose output");
    printf("\n\n");

    exit (EXIT_FAILURE);
}

